#include "crestSoftKeys.h"
#include <linux/input/mt.h>

#define MAX_KEY_ROWS    5
#define USED_KEY_ROWS   5
#define MAX_KEY_COLS    1
#define USED_KEY_COLS   1
#define MAX_KEY_NUM     (MAX_KEY_ROWS * MAX_KEY_COLS)
#define ROW_SHIFT       3
#define NUM_OF_HARDKEYS 5

extern int uiVbuttonIntfInit(struct crestSoftKeys_ctrl_t* ctrl);
struct mutex *keyStateMutex; //mutex lock for gVbKeyStates

extern struct UiHalTouch uiTouchData[];
extern int uiMultiTouch;

#define CREST_SOFT_KEYS_TOUCH_SPACE_SCREEN 1
char crestSoftKeys_getTouchSpace(struct softkey_area *key_area, struct crest_xydata_t touch);
int crestSoftKeys_getVirtualButton(struct softkey_area *key_area, u16 y);
int crestSoftKeys_processVirtualButton(struct crestSoftKeys_ctrl_t *ctrl, struct crest_xydata_t touch);

static ssize_t crestSoftKeys_vkey_show_event(struct device *dev,struct device_attribute *attr, const char *buf)
{
	return 0;
}

static DEVICE_ATTR(event, 				S_IRUGO,			crestSoftKeys_vkey_show_event,				NULL);

static struct attribute *crestSoftKeys_vkey_attributes[] = {
	&dev_attr_event.attr,
	NULL
};

static struct attribute_group crestSoftKeys_vkey_attribute_group = {
    .attrs = crestSoftKeys_vkey_attributes
};

/**
 * Emit the point via multitouch protocol B.
 */
int emitPointProtocolB(struct input_dev *ts_dev, int touchId, struct crest_xydata_t touch)
{
	//dev_info(&ts->client->dev, "%s: tch=%i ev=%i x=%3i y=%3i\n", __FUNCTION__, touchId, state, x, y);
	// Populate Crestron uiHal
	if(touchId < UIHAL_MAX_TOUCH_POINTS_SUPPORTED) {
		uiTouchData[touchId].touchId=touchId;
		uiTouchData[touchId].pressState=touch.state;
		uiTouchData[touchId].xPosition=touch.pos[0];
		uiTouchData[touchId].yPosition=touch.pos[1];
	}

	if (!g_disableInputEvents) {
		input_mt_slot(ts_dev, touchId);
		input_mt_report_slot_state(ts_dev, MT_TOOL_FINGER, touch.state);
		if(touch.state)
		{
			input_report_abs(ts_dev, ABS_MT_POSITION_X, touch.pos[0]);
			input_report_abs(ts_dev, ABS_MT_POSITION_Y, touch.pos[1]);
			//input_mt_report_pointer_emulation(ts_dev, 1);
		}
	}
	input_sync(ts_dev);
	return 0;
}

/**
 * Extracted from Read_xxx_1050 function above and repurposed for other touch screens.
 * Outputs points and key-up/key-down events to Android if not in the virtual
 * key area.
 */
void crestSoftKeys_processTouchData(struct crestSoftKeys_ctrl_t *ctrl, int touchId, struct crest_xydata_t touch, bool forceSend)
{
	struct crest_xydata_t *last = &ctrl->prv_tch[touchId];
	char touchSpace = crestSoftKeys_getTouchSpace(&ctrl->softkeys, touch);

	if(touch.pos[1] > ctrl->maxy - ctrl->y_bottom_cap)
		touch.pos[1] = ctrl->maxy - ctrl->y_bottom_cap;

	//printk(KERN_INFO "%s: id=%i old_space=%i new_space=%i\n", __FUNCTION__, touchId, last->state, touchSpace);
	bool transitioned = (touchSpace != last->state);

	if(unlikely(touchId > CRESTRON_MAX_NUM_TOUCHES))
	{
		printk(KERN_ERR "%s: touchId = %i\n", __FUNCTION__, touchId);
		return;
	}

	/* Always send transition events. Detect if we've crossed boundaries from
	 * one space to the other. If so, send touch-up in the old space and
	 * touch-down in the new. */
	if(unlikely(transitioned))
	{
		//dev_info(ts_dev, "Transitioned\n");
		if(last->state) {
			struct crest_xydata_t upTouch = *last;
			upTouch.state = FT_NTCH;
			if(last->state > CREST_SOFT_KEYS_TOUCH_SPACE_SCREEN) { // If it was on a virtual key previously.
				crestSoftKeys_processVirtualButton(ctrl, upTouch);
			}
			else {
				emitPointProtocolB(ctrl->ts_dev, touchId, upTouch);
			}
		}
	}
	if(likely(transitioned || forceSend)) /* Otherwise, only send forced events. */
	{
		if(touchSpace > CREST_SOFT_KEYS_TOUCH_SPACE_SCREEN) {
			crestSoftKeys_processVirtualButton(ctrl, touch);
		}
		else {
			emitPointProtocolB(ctrl->ts_dev, touchId, touch);
		}
	}
	if(unlikely(transitioned))
	{
		if(last->state == CREST_SOFT_KEYS_TOUCH_SPACE_SCREEN || touchSpace == CREST_SOFT_KEYS_TOUCH_SPACE_SCREEN)
			sysfs_notify(&ctrl->ts_dev->dev.kobj, NULL, "event");
		if(last->state > CREST_SOFT_KEYS_TOUCH_SPACE_SCREEN || touchSpace > CREST_SOFT_KEYS_TOUCH_SPACE_SCREEN)
			sysfs_notify(&ctrl->key_dev->dev.kobj, NULL, "event");
	}
	touch.state = touchSpace;
	*last = touch;
}

/**
 * Creates and initializes input device to send key events.
 */
int crestSoftKeys_init(struct crestSoftKeys_ctrl_t *ctrl)
{
	int virt_key;
	int retval = 0;

	/* Create the input device and register it. */
	ctrl->key_dev = input_allocate_device();
	if (!ctrl->key_dev)
	{
		retval = -ENOMEM;
		printk(KERN_ERR "%s() - ERROR: Could not allocate input device.\n", __FUNCTION__);
		goto error_free_device;
	}
	ctrl->key_dev->name = "touchVKey";

	/* Setup functionality */
	set_bit(EV_KEY, ctrl->key_dev->evbit);
	for(virt_key=0; virt_key < SOFTKEY_NUM_VIRT_BUTTON; ++virt_key) {
		set_bit(ctrl->softkeys.virt_btn[virt_key].keycode, ctrl->key_dev->keybit);
	}
	set_bit(KEY_F14, ctrl->key_dev->keybit);

	retval = input_register_device(ctrl->key_dev);
	if(retval) {
		printk(KERN_ERR "%s() - ERROR: Could not register input softkey device.\n", __FUNCTION__);
		goto error_free_device;
	}
	retval = uiVbuttonIntfInit(ctrl);
	if(retval) {
		printk(KERN_ERR "%s() - ERROR: Failed to initialize uiVbutton interface.\n", __FUNCTION__);
		goto error_deregister_device;
	}

	retval = sysfs_create_group(&ctrl->key_dev->dev.kobj, &crestSoftKeys_vkey_attribute_group);
	if (retval)
	{
		printk(KERN_ERR "%s() - ERROR: sysfs_create_group() failed: %d\n", __FUNCTION__, retval);
	}
	else
	{
		printk(KERN_INFO "%s() - sysfs_create_group() succeeded.\n", __FUNCTION__);
	}

	goto success;
error_deregister_device:
	/* Do not call input_free_device if unregister is called. */
	input_unregister_device(ctrl->key_dev);
	ctrl->key_dev = NULL;

error_free_device:
	if(ctrl->key_dev)
	{
		input_free_device(ctrl->key_dev);
		ctrl->key_dev = NULL;
	}
success:
	return retval;
}

void printKeyArea(struct softkey_area *key_area)
{
	int i=0;
	printk(KERN_INFO "key_area=%p has_virt_btn=%i startx=%i radius=%i\n",
			key_area, key_area->has_virt_btn, key_area->virt_btn_startx, key_area->virt_btn_radius);
	for(; i < SOFTKEY_NUM_VIRT_BUTTON; ++i) {
		printk(KERN_INFO " virt_btn[%i].center = %i\n", i, key_area->virt_btn[i].centery);
	}
}

int crestSoftKeys_getVirtualButton(struct softkey_area *key_area, u16 y)
{
	int virt_key; // Index of the virtual keys we've defined
	int vButton = -1; // The virtual key we think has activity.

	for(virt_key=0; virt_key < SOFTKEY_NUM_VIRT_BUTTON; ++virt_key) {
    	// Calculate edges around button where it is still considered the button
    	if( (y > key_area->virt_btn[virt_key].centery - key_area->virt_btn_radius) &&
            (y < key_area->virt_btn[virt_key].centery + key_area->virt_btn_radius)) {
            vButton = virt_key;
            break;
        }
    }
    return vButton;
}

/**
 * Test to see if the touch is in the virtual keyspace.
 */
inline bool crestSoftKeys_isInVirtKeySpace(struct softkey_area *key_area, u16 x)
{
	return (key_area->has_virt_btn &&
			(x > key_area->virt_btn_startx));
}

char crestSoftKeys_getTouchSpace(struct softkey_area *key_area, struct crest_xydata_t touch)
{
	char touchSpace = touch.state; // Assume in the touch area.
	char vkey = 0;
	if(touch.state && crestSoftKeys_isInVirtKeySpace(key_area, touch.pos[0])) {
		/* Use 2-6 to represent each virtual key. getVirtualButton returns 0-4 */
		vkey = crestSoftKeys_getVirtualButton(key_area, touch.pos[1]);
		if(vkey < 0) {
			return 0;
		}
		touchSpace = vkey + CREST_SOFT_KEYS_TOUCH_SPACE_SCREEN + 1;
	}
	return touchSpace;
}

int crestSoftKeys_processVirtualButton(struct crestSoftKeys_ctrl_t *ctrl, struct crest_xydata_t touch)
{
	//printk(KERN_INFO "%s: touch = (%4i,%4i) state = %i\n", __FUNCTION__, touch.pos[0], touch.pos[1], touch.state);
    int vButton = crestSoftKeys_getVirtualButton(&ctrl->softkeys, touch.pos[1]);
    if(vButton < 0) {
    	// No virtual button in this location
    	return 0;
    }

	if (g_virt_dbg_keys) {
		// Send the buttons up to android only if DEBUG keys are set through /proc. otherwise no keys go to Android.
		input_report_key(ctrl->key_dev, ctrl->softkeys.virt_btn[vButton].keycode, !!touch.state);
		input_sync(ctrl->key_dev);
	}
#if 0
	if (!g_disableInputEvents) {
		// UIHAL still needs to get a key event in order to read the proc
		input_report_key(ctrl->key_dev, KEY_F14, !!touch.state);
	}
#endif
    mutex_lock(keyStateMutex);  //mutex lock for gVbKeyStates
    // Now send the buttons to the uiHAL.
    if (!touch.state) {
		gVbKeyStates &= ~( 0x0001 << vButton );
    } else {
		gVbKeyStates |= ( 0x0001 << vButton );
    }
	mutex_unlock(keyStateMutex);

    return 0;
}

